/*===========================================================*/
/* GPS Routines				                     */
/* (c) MSP			                     	     */
/*===========================================================*/

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this package; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA. 


#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "common.h"
#include "gps.h"


t_GPSinfo GPSinfo;

uint8_t buff[8];


/*------------------------------------------------*/
/* Find next comma                                */
/* Returns number of data before next comma       */
/*------------------------------------------------*/
uint8_t next_comma(uint8_t *str)
{
   int8_t i;

   i = 0;
   while ((str[i] != 0x00) && (str[i] != ','))
      i++;
     
   return i;
}


/*------------------------------------------------*/
/* Parse GGA data                                 */
/*------------------------------------------------*/
void gps_process_GGA(uint8_t recstr[])
{   
   uint8_t i, j;

   // initial string ($GPGGA)
   i = next_comma(recstr);

   // UTC Time
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
   {
      strncpy(buff, recstr, 2);
      buff[2] = 0;
      GPSinfo.time.h = atoi(buff);
      strncpy(buff, recstr+2, 2);
      GPSinfo.time.m = atoi(buff);
      GPSinfo.time.s = atoi(recstr+4);
   }

   // lattitude [ddmm.mmmmm]
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
   {
      j = 0;
      while (recstr[j] != '.')
	 j++;
      strncpy(buff, recstr, j-2);
      buff[j-2] = 0;
      GPSinfo.lattitude = atoi(buff) * (int32_t)100000;
      strncpy(buff, recstr+j-2, 2);
      strncpy(buff+2, recstr+j+1, i-j-1);
      buff[i-j+1] = 0;
      GPSinfo.lattitude += (int32_t)strtol(buff, NULL, 10) / 6;
   }

   // lattitude N/S
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
      if(*recstr == 'S')
	 GPSinfo.lattitude *= -1;

   // longitude [ddmm.mmmmm]
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
   {
      j = 0;
      while (recstr[j] != '.')
	 j++;
      strncpy(buff, recstr, j-2);
      buff[j-2] = 0;
      GPSinfo.longitude = atoi(buff) * (int32_t)100000;
      strncpy(buff, recstr+j-2, 2);
      strncpy(buff+2, recstr+j+1, i-j-1);
      buff[i-j+1] = 0;
      GPSinfo.longitude += (int32_t)strtol(buff, NULL, 10) / 6;
   }

   // longitude E/W
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
      if(*recstr == 'W')
	 GPSinfo.longitude *= -1;

   // quality indicator
   recstr += i+1;
   i = next_comma(recstr);
   
   // Number of sattelites
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
      GPSinfo.satnum = atoi(recstr);

   // Horizontal dilution of position
   recstr += i+1;
   i = next_comma(recstr);
   
   // Antenna altitude above/below mean sea level (geoid) 
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
      GPSinfo.altitude = atoi(recstr);

}

/*------------------------------------------------*/
/* Parse VTG data                                 */
/*------------------------------------------------*/
void gps_process_VTG(uint8_t recstr[])
{   
   uint8_t i, j;

   // initial string ($GPVTG)
   i = next_comma(recstr);

   // track made good
   recstr += i+1;
   i = next_comma(recstr);

   // fixed text 'T' - not checked
   recstr += i+1;
   i = next_comma(recstr);

   // not used
   recstr += i+1;
   i = next_comma(recstr);

   // not used
   recstr += i+1;
   i = next_comma(recstr);

   // speed in knots
   recstr += i+1;
   i = next_comma(recstr);

   // fixed text 'N' - not checked
   recstr += i+1;
   i = next_comma(recstr);
   
   // speed in km/h
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
   {
      j = 0;
      while (recstr[j] != '.')
	 j++;
      strncpy(buff, recstr, j);
      buff[j] = 0;
      GPSinfo.speed = atoi(buff);
   }

   // fixed text 'K'
   recstr += i+1;
   i = next_comma(recstr);
   if (i!=0)
   {
      if (recstr[0] != 'K')
         GPSinfo.speed = 0;
   }
}


/*------------------------------------------------*/
/* Distance calculation                           */
/*------------------------------------------------*/
int32_t gps_distance(int32_t lattitude1, int32_t lattitude2, 
         	     int32_t longitude1, int32_t longitude2)
{
   double op1, sqr1;
   double op2, sqr2;
   int32_t res;

   op1 = 69.1*(double)(lattitude1 - lattitude2);
   op2 = 69.1*(double)(longitude1 - longitude2);
   sqr1 = op1*op1;
   sqr2 = op2*op2;

   res = (int32_t)(sqrt(sqr1+sqr2)/100);
   return res;
}

