/*===========================================================*/
/* GPS POI Routines 			                     */
/* (c) MSP			                     	     */
/*===========================================================*/

// This is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this package; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
 

#include <string.h>
#include <stdlib.h>
#include <avr/pgmspace.h>
#include "common.h"
#include "gps.h"
#include "poi.h"
#include "poi_data.h"
#include "sound.h"



typedef struct
{
   uint32_t size;
   int32_t  longitude;
   int32_t  lattitude;
} t_POIfield;


t_POI POI;

uint8_t  poi_on = 0;

/*------------------------------------------------*/
/* POI main process                               */
/*------------------------------------------------*/
void poi(int32_t longitude, int32_t lattitude)
{
   uint16_t mindist_w;

   // find POI
   nearest_poi(longitude, lattitude);

   // calculate min. warning distance for 10 sec., but at least 120m
   //mindist_w = (GPSinfo.speed * 100) / 36;
   mindist_w = GPSinfo.speed / 3;
   if (mindist_w < 120)
      mindist_w = 120;
   
   // start warning
   if ((POI.distance < mindist_w) && (poi_on == 0))
   {
      PORTE |= LED_RD;
      poi_on = POI.type;
      switch (POI.type)
      {
	 case(1):
	    sound_on(1, 20, 10);
	    break;
	 case(2):
	    sound_on(2, 20 , 10);
	    break;
	 case(3):
	    sound_on(2, 30 , 10);
	    break;
	 case(4):
	    sound_on(3, 20 , 10);
	    break;
	 case(5):
	    sound_on(3, 30 , 10);
	    break;
      	 default:
	    sound_on(1, 40, 10);
	    break;
      }
   }

   // stop warning
   if ((POI.distance > (mindist_w+30)) && (poi_on != 0))
   {
      PORTE &= !(LED_RD);
      poi_on = 0;
   }
}


/*------------------------------------------------*/
/* Find nearest POI                               */
/*------------------------------------------------*/
void nearest_poi(int32_t longitude, int32_t lattitude)
{
   int32_t distance;
   int32_t mindist;
   int32_t mindist_ptr;
   uint8_t mindist_type;
   
   uint16_t EEPROM_ptr;
   uint8_t  POIdata_type;
   uint32_t POIdata_ptr, POIdata_ptr_e;
   uint8_t  last_poi;
   
   t_POIfield POIfield;
   
   
   mindist = INT32_MAX;
   mindist_type = 0;
   
   EEPROM_ptr = 0;
   last_poi = 1;

   // loop through POI sets
   do
   {
      POIdata_type = eeprom_read_byte((uint8_t*)EEPROM_ptr);
      EEPROM_ptr += 1;
      last_poi = ((POIdata_type == 0) || (POIdata_type == 0xFF));

      if (last_poi == 0)
      {
         // address of POI data (32->24bit)
	 POIdata_ptr = eeprom_read_word((uint16_t*)EEPROM_ptr);
	 EEPROM_ptr += 2;
	 POIdata_ptr += ((uint32_t)eeprom_read_byte((uint8_t*)EEPROM_ptr) << 16) + 1;
	 EEPROM_ptr += 2;
	 // size of POI data (32->24 bit)
	 POIdata_ptr_e = eeprom_read_word((uint16_t*)EEPROM_ptr);
	 EEPROM_ptr += 2;
	 POIdata_ptr_e += ((uint32_t)eeprom_read_byte((uint8_t*)EEPROM_ptr) << 16) + POIdata_ptr;
	 EEPROM_ptr += 2;
	 
	 // find nearest POI
	 while (POIdata_ptr < POIdata_ptr_e)
	 {
	    
	    // read POI data
	    POIfield.size = pgm_read_dword_far(POIdata_ptr);
	    POIfield.longitude = pgm_read_dword_far(POIdata_ptr + 4);
	    POIfield.lattitude = pgm_read_dword_far(POIdata_ptr + 8);

	    // calculate distance
	    distance = gps_distance(POIfield.lattitude, GPSinfo.lattitude,
				    POIfield.longitude, GPSinfo.longitude);

	    if (distance < mindist)
	    {
	       mindist = distance;
	       mindist_ptr = POIdata_ptr;
	       mindist_type = POIdata_type;
	    }

	    POIdata_ptr += POIfield.size;
	 }
      }
   } while (last_poi == 0);
   
   // store nearest POI
   POI.distance = mindist;
   POI.name = mindist_ptr + 12;
   POI.type = mindist_type;
   POI.longitude = pgm_read_dword_far(mindist_ptr + 4);
   POI.lattitude = pgm_read_dword_far(mindist_ptr + 8);
}

